package nl.ivojonker.icn.configuration;

import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Properties;
import java.util.logging.Logger;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
 
/**
 * Manages the creation and retrieval of properties from the (within .jar) configured datasource/tablename.
 * 
 * Requires a websphere-application server context as it looks up datasources using the {@link com.ibm.websphere.naming.WsnInitialContextFactory } 
 * 
 * @author nl.ivojonker
 *
 */
public class Configuration {

	private static Logger logger = Logger.getLogger(Configuration.class.getCanonicalName());
	
	private static String DATASOURCEJNDI;
	private static String CONFIGTABLENAME;
	private static String CREATETABLEQUERY;
	
	private static Configuration _instance;

	public static final String DB_COL_KEY="id";
	public static final String DB_COL_VALUE="value";
	public static final String DB_COL_EXPOSEFRONTEND="expose";
	public static final String DB_COL_EVALFRONTEND="eval"; 
	public static final String DB_COL_DESCRIPTION="description"; 


	private Properties flatConfiguration=new Properties();
	private ArrayList<ConfigurationEntry> config;

	/**
	 * When creating the instance, search the config.properties for the datasource/table name to use.
	 */
	private Configuration(){
		try{
			InputStream propertyFile = Configuration.class.getClassLoader().getResourceAsStream("config.properties");
			Properties properties = new Properties();
			properties.load(propertyFile);

			DATASOURCEJNDI=properties.getProperty("DATASOURCEJNDI");
			CONFIGTABLENAME=properties.getProperty("CONFIGTABLENAME");
			CREATETABLEQUERY=properties.getProperty("CREATETABLEQUERY");
		}catch(Exception any){
			throw new ConfigException("Failure while trying to read plugin.jar!config.properties -> "+any.getMessage(),any);
		}
		
		flatConfiguration=new Properties();
	}

	
	/**
	 * Gets an instance from the database; will use cache. 
	 * @param useCache Forces a database roundtrip; updates the current classloader's cache.
	 * @return
	 * @throws SQLException
	 */
	public static Configuration getInstance(boolean useCache) throws SQLException{
		if (_instance==null) {
			_instance=new Configuration();
			useCache=false;
		}
		
		if (!useCache){
			_instance.refreshConfigFromDB();
		}
		return _instance;
	}
	
	/**
	 * Gets an instance from the database; will use cache. 
	 * @return
	 * @throws SQLException
	 */
	public static Configuration getInstance() throws SQLException{
		return getInstance(true);
	}
	
	/**
	 * Always returns a functional ConfigurationObject, even if exceptions occur. 
	 * Exceptions aren't logged; Advised to use getProperty(key,defaultValueIfNullOrNotPresent).
	 * @param useCache
	 * @return
	 */
	public static Configuration getInstanceSupressExceptions(boolean useCache) {
		try{
			getInstance(useCache);
		}catch(Exception ignoreAll){
			;
		}
		return _instance;
	}
	
		
	
	/**
	 * Queries the database.
	 * @throws SQLException
	 */
	private synchronized void refreshConfigFromDB() throws SQLException {
		ArrayList<ConfigurationEntry> rawConfig = new ArrayList<ConfigurationEntry>();
		Properties kvConfig = new Properties(); 
		
		Connection dsConnection=null;
		Statement stmt=null;
		ResultSet rs=null;
		
		String query = "select * from "+CONFIGTABLENAME;

		try{//Retrieve ICN plugin configuration
			logger.fine("Opening connection to navigator plugin DS");
			dsConnection = getConfigDS().getConnection();
			stmt = dsConnection.createStatement();
			rs = stmt.executeQuery(query);
			while(rs.next()){
				String key = rs.getString(DB_COL_KEY);
				String value = rs.getString(DB_COL_VALUE);
				String eval = rs.getString(DB_COL_EVALFRONTEND);
				String expose = rs.getString(DB_COL_EXPOSEFRONTEND);
				String description = rs.getString(DB_COL_DESCRIPTION);
				kvConfig.put(rs.getString(DB_COL_KEY),rs.getString("value"));
				
				rawConfig.add(new ConfigurationEntry(key, value, Boolean.valueOf(eval), Boolean.valueOf(expose),description));
			}
		}finally{
			logger.fine("Closing connection to navigator plugin DS");
			try{rs.close();}catch(Exception ignoreAll){}
			try{stmt.close();}catch(Exception ignoreAll){}
			try{dsConnection.close();}catch(Exception ignoreAll){
				logger.warning("Error closing connection to navigator plugin DS. Please mind connectionpool!");
			}
		}
		
		this.flatConfiguration.clear();
		this.flatConfiguration.putAll(kvConfig);
		this.config=rawConfig;
	}



	/**
	 * Looks up the datasource in the websphere context.
	 * @param JNDIName
	 * @return
	 */
	private DataSource getConfigDS(){
		try{
			logger.fine("Create websphere naming InitialContextFactory");
			Hashtable<String, String> env = new Hashtable<String, String>();
			env.put(Context.INITIAL_CONTEXT_FACTORY, "com.ibm.websphere.naming.WsnInitialContextFactory");

			Context ctx = new InitialContext(env);

			logger.finest("Lookup "+DATASOURCEJNDI);

			DataSource ods = (DataSource) ctx.lookup(DATASOURCEJNDI); 
			if(ods != null)
				logger.finest("Lookup done ");
			else
				throw new NamingException("Failed lookup (returned null)");
			return ods;
		}catch(Exception any){
			logger.severe("Failed to lookup datasource for configuration, JNDI:"+DATASOURCEJNDI);
			throw new ConfigException("Failed to lookup datasource for configuration, JNDI:"+DATASOURCEJNDI+", reason: "+any.getMessage());
		}
	}
	
	/**
	 * Persists a set of key/values to the database. Deleting the previous settings.
	 * @param props
	 * @throws SQLException
	 */
	protected synchronized void persist(ArrayList<ConfigurationEntry> props) throws SQLException{
		Connection dsConnection=null;
		
		try{//Retrieve ICN plugin configuration
			logger.fine("Opening connection to "+DATASOURCEJNDI);
			dsConnection = getConfigDS().getConnection();
			dsConnection.setAutoCommit(true);
			logger.fine("Persisting configuration to "+DATASOURCEJNDI);
			Statement stmtClear = dsConnection.createStatement();
			stmtClear.executeUpdate("delete from "+CONFIGTABLENAME);
			stmtClear.close();
			
			for (ConfigurationEntry confProperty : props) {
				logger.finer(String.format("Persisting property %1$s with value %2$s",confProperty.getKey(),confProperty.getValue()));
				//property by property, not to assume sql-92 support
				PreparedStatement stmtInsert = dsConnection.prepareStatement("insert into "+CONFIGTABLENAME+"("+DB_COL_KEY+","+DB_COL_VALUE+","+DB_COL_EVALFRONTEND+","+DB_COL_EXPOSEFRONTEND+","+DB_COL_DESCRIPTION+") values(?,?,?,?,?)");
				stmtInsert.setString(1, confProperty.getKey());
				stmtInsert.setString(2, confProperty.getValue());
				stmtInsert.setString(3, confProperty.getEvaluate().toString());
				stmtInsert.setString(4, confProperty.getExpose().toString());
				stmtInsert.setString(5, confProperty.getDescription());
				stmtInsert.executeUpdate();
				stmtInsert.close();
			}
		}finally{
			logger.fine("Closing connection to navigator plugin DS"); 
			try{dsConnection.close();}catch(Exception ignoreAll){
				logger.warning("Error closing connection to navigator plugin DS. Please mind connectionpool!");
			}
		}
	}
	
	public synchronized void createTable() throws SQLException{
		Connection dsConnection=null;
		Statement stmt = null;
		try{
			String sqlCreateTable = String.format(CREATETABLEQUERY,
					CONFIGTABLENAME,
					DB_COL_KEY,
					DB_COL_VALUE, 
					DB_COL_EVALFRONTEND,
					DB_COL_EXPOSEFRONTEND,
					DB_COL_DESCRIPTION
					);
			dsConnection=getConfigDS().getConnection();
			dsConnection.setAutoCommit(true);
			stmt=dsConnection.createStatement();
			stmt.executeUpdate(sqlCreateTable);
		}finally{
			try{ dsConnection.close(); }catch(Exception ignore){;}
			try{ stmt.close(); }catch(Exception ignore){;}
		}
	}


	/**
	 * Retrieves a property from the instance
	 * @param canonicalPropertyName
	 * @param valueIfNullOrNotPresent
	 * @return
	 */
	public synchronized String getProperty(String canonicalPropertyName, String valueIfNullOrNotPresent){
		if (!flatConfiguration.containsKey(canonicalPropertyName))
			logger.severe("Requested property nog configured in central configuration instance: "+canonicalPropertyName);
		return flatConfiguration.getProperty(canonicalPropertyName, valueIfNullOrNotPresent);
	}
	
	/**
	 * Retrieves a property from the instance; will result null if property not present.
	 * @param canonicalPropertyName
	 * @return
	 */
	public synchronized String getProperty(String canonicalPropertyName){
		return getProperty(canonicalPropertyName,null);
	}
	
	
	protected ArrayList<ConfigurationEntry> getRawConfig(){
		return this.config;
	}


	/** 
	 * drops the configuration table.
	 * @throws SQLException
	 */
	public void dropTable() throws SQLException {
		Connection dsConnection=null;
		Statement stmt = null;
		try{
			String sqlDropTable = "drop table "+CONFIGTABLENAME;
			dsConnection=getConfigDS().getConnection();
			dsConnection.setAutoCommit(true);
			stmt=dsConnection.createStatement();
			stmt.executeUpdate(sqlDropTable);
		}finally{
			try{ dsConnection.close(); }catch(Exception ignore){;}
			try{ stmt.close(); }catch(Exception ignore){;}
		}
	}

	
	
}
